package models.projects;

import com.google.gson.annotations.Expose;
import models.*;
import models.contacts.Customer;
import models.contacts.Person;
import org.hibernate.annotations.Type;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import play.Logger;
import play.db.jpa.Model;
import play.mvc.Router;

import javax.persistence.*;
import java.lang.reflect.Array;
import java.util.*;
import java.util.stream.Collector;
import java.util.stream.Collectors;

/**
 * Created by dom on 5/12/2018.
 */
@Entity
public class Project extends Model implements Event {
    public static final String DD_MM_YYYY = "dd-MM-yyyy";

    @Searchable
    public String name;

    @Searchable
    public String description;

    public String status;

    @OneToMany
    public List<Deliverable> deliverables = new ArrayList<>();

    @Embedded
    public ProjectFinancialInfo financialInfo = new ProjectFinancialInfo();

    @Expose
    @Column
    @Type(type = "org.jadira.usertype.dateandtime.joda.PersistentDateTime")
    public DateTime projectStartDate;

    @Expose
    @Column
    @Type(type = "org.jadira.usertype.dateandtime.joda.PersistentDateTime")
    public DateTime projectDeliverDate;

    @OneToMany
    public List<DBFile> fileList = new ArrayList<>();

    @ManyToMany
    public Set<AppUser> userSet = new TreeSet<>();

    @OneToMany
    public List<Share> shares = new ArrayList<>();

    @ManyToOne
    public Customer customer;

    @ManyToOne
    public Company company;

    @OneToMany(mappedBy = "project")
    public List<Task> tasks = new ArrayList<>();


    public Document coverImage;

    public Document contract;

    @Column
    @Type(type = "org.jadira.usertype.dateandtime.joda.PersistentDateTime")
    public DateTime createDate;

    @ManyToOne
    public AppUser uploader;

    public Project() {

    }

    public String getProjectStartDateString() {
        if (this.projectStartDate == null) {
            return "";
        }
        DateTimeFormatter fmt = DateTimeFormat.forPattern(DD_MM_YYYY);
        return fmt.print(projectStartDate);
    }

    public String getProjectDeliverDateString() {
        if (this.projectDeliverDate == null) {
            return "";
        }
        DateTimeFormatter fmt = DateTimeFormat.forPattern(DD_MM_YYYY);
        return fmt.print(projectDeliverDate);
    }

    public boolean isPersonBindedtoProject(Long userId) {
        AppUser user = AppUser.findById(userId);
        Share share = Share.find("select s from Share s where user =:usr and project =:proj")
                .setParameter("usr", user)
                .setParameter("proj", this).first();
        if (share == null) {
            return false;
        } else {
            return true;
        }
    }

    public Integer getProjectSharedNumber() {
        List<Share> shares = Share.find("select s from Share s where project =:proj")
                .setParameter("proj", this).fetch();

        return shares.size();
    }

    public Long getActiveToDoCount()
    {
        List<Task> tasks = this.tasks.stream().filter(t -> t.status.equals("0")).collect(Collectors.toList());
        return  tasks.stream().filter((t -> t.getClass().getSimpleName().equals("Task"))).count();
    }

    @Override
    public DateTime getStartDate() {
        return projectStartDate;
    }

    @Override
    public String getTitle() {
        return name;
    }

    public String toString(){
        return getTitle();
    }

    @Override
    public String getCalendarContent() {
        return "";
    }


    @Override
    public String getUrl() {
        Map<String, Object> map = new HashMap<String, Object>();
        map.put("id", this.id);
        return Router.getFullUrl("Projects.addProject", map);
    }

    @Override
    public String getUploadedBy() {
        return uploader.userName;
    }

    @Override
    public JSONEvent getEvent() {
        JSONEvent event = new JSONEvent(getStartDate().toDate(), getTitle(), getCalendarContent(), "#49742c", this.getUrl(), this.getUploadedBy(), this.getClass());
        return event;
    }
}

